<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';

// Permissions check
if (!hasPermission('view_reports') && !hasPermission('export_reports')) {
    $_SESSION['error'] = "You don't have permission to view teller logs.";
    header('Location: dashboard.php');
    exit;
}

// Get tellers (users who have made sales)
$tellers = $pdo->query("SELECT DISTINCT u.id, u.full_name, u.username FROM users u JOIN sales s ON u.id = s.user_id ORDER BY u.full_name, u.username")->fetchAll();

// Get filters
$selected_teller = $_GET['teller'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Build query for all sales
$query = "
    SELECT s.created_at, u.full_name, u.username, p.name AS product_name, si.quantity, si.total_price, c.name AS customer_name, s.payment_method
    FROM sales s
    JOIN users u ON s.user_id = u.id
    JOIN sale_items si ON s.id = si.sale_id
    JOIN products p ON si.product_id = p.id
    LEFT JOIN customers c ON s.customer_id = c.id
    WHERE DATE(s.created_at) BETWEEN ? AND ?
";
$params = [$start_date, $end_date];
if ($selected_teller) {
    $query .= " AND u.id = ?";
    $params[] = $selected_teller;
}
$query .= " ORDER BY s.created_at DESC";
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$sales = $stmt->fetchAll();

// Calculate total sales and credit sales made
$total_sales = 0;
$total_credit_sales = 0;
foreach ($sales as $sale) {
    $total_sales += $sale['total_price'];
    if (isset($sale['payment_method']) && strtolower($sale['payment_method']) === 'credit') {
        $total_credit_sales += $sale['total_price'];
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teller Logs - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #e0e7ff 0%, #f8fafc 100%); min-height: 100vh; }
        .card-modern {
            border-radius: 22px;
            box-shadow: 0 8px 32px rgba(37,99,235,0.13), 0 1.5px 6px rgba(0,0,0,0.03);
            border: none;
            background: rgba(255,255,255,0.92);
            backdrop-filter: blur(6px);
        }
        .summary-card {
            border-radius: 18px;
            background: linear-gradient(90deg, #6366f1 0%, #38bdf8 100%);
            color: #fff;
            box-shadow: 0 4px 24px #6366f1aa;
            padding: 1.5rem 2rem;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }
        .summary-card .icon {
            font-size: 2.5rem;
            margin-right: 1rem;
        }
        .summary-card .value {
            font-size: 2.2rem;
            font-weight: 700;
        }
        .summary-card .label {
            font-size: 1.1rem;
            font-weight: 500;
            opacity: 0.95;
        }
        .credit-card {
            border-radius: 18px;
            background: linear-gradient(90deg, #f59e42 0%, #fbbf24 100%);
            color: #fff;
            box-shadow: 0 4px 24px #f59e42aa;
            padding: 1.5rem 2rem;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }
        .credit-card .icon {
            font-size: 2.5rem;
            margin-right: 1rem;
        }
        .credit-card .value {
            font-size: 2.2rem;
            font-weight: 700;
        }
        .credit-card .label {
            font-size: 1.1rem;
            font-weight: 500;
            opacity: 0.95;
        }
        .glass-bg {
            background: rgba(255,255,255,0.85);
            border-radius: 18px;
            box-shadow: 0 2px 16px rgba(59,130,246,0.10);
            padding: 2rem 2.5rem 2rem 2.5rem;
            margin-bottom: 2.5rem;
        }
        @media (max-width: 991px) {
            .glass-bg { padding: 1.2rem 0.7rem; }
            .summary-card, .credit-card { flex-direction: column; align-items: flex-start; padding: 1.2rem 1rem; }
        }
        @media print {
            .no-print { display: none !important; }
            .sidebar-modern, .footer, .navbar, .topbar, .card-modern, .glass-bg, .summary-card, .credit-card { box-shadow: none !important; }
            body { background: #fff !important; }
            .container-fluid { padding: 0 !important; }
            .table { font-size: 0.95em; }
            /* Hide all elements except summary cards and table */
            .sidebar-modern, .topbar, .footer, h2, form, .navbar, nav, header, .navbar-expand-lg, .navbar-brand, .navbar-nav, .dropdown, .user-dropdown, .dropdown-menu, .dropdown-item, .btn-group, .btn-outline-light, .navbar-toggler, .navbar-collapse { display: none !important; }
            .col-md-10 { margin-left: 0 !important; padding: 0 !important; }
            .card-body { padding: 0 !important; }
            .glass-bg { background: transparent !important; box-shadow: none !important; }
            .card-modern { background: transparent !important; box-shadow: none !important; }
            /* Show only summary cards and table */
            .summary-card, .credit-card, .table-responsive { display: block !important; }
            .row.g-3 { margin: 0 !important; }
            .col-lg-6 { width: 50% !important; float: left !important; }
            .mb-4 { margin-bottom: 1rem !important; }
            /* Additional hiding for any remaining elements */
            .py-4 { padding: 0 !important; }
            .ms-sm-auto { margin-left: 0 !important; }
            .px-md-4 { padding: 0 !important; }
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <div class="col-md-10 ms-sm-auto px-md-4">
            <?php include '../includes/topbar.php'; ?>
            <h2 class="mb-4 mt-3 fw-bold" style="letter-spacing:0.01em;"><i class="fas fa-user-clock me-2"></i>Teller Logs</h2>
            <div class="row g-3 mb-4">
                <div class="col-lg-6">
                    <div class="summary-card">
                        <span class="icon"><i class="fas fa-coins"></i></span>
                        <div>
                            <div class="value"><?php echo number_format($total_sales, 2); ?></div>
                            <div class="label">Total Sales Made<?php if ($selected_teller) { echo ' by Teller'; } ?><?php if ($start_date || $end_date) { echo ' (' . htmlspecialchars($start_date) . ' to ' . htmlspecialchars($end_date) . ')'; } ?></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="credit-card">
                        <span class="icon"><i class="fas fa-credit-card"></i></span>
                        <div>
                            <div class="value"><?php echo number_format($total_credit_sales, 2); ?></div>
                            <div class="label">Credit Sales<?php if ($selected_teller) { echo ' by Teller'; } ?><?php if ($start_date || $end_date) { echo ' (' . htmlspecialchars($start_date) . ' to ' . htmlspecialchars($end_date) . ')'; } ?></div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="card card-modern glass-bg mb-4">
                <div class="card-body">
                    <div class="mb-3 d-flex justify-content-end">
                        <button type="button" class="btn btn-secondary no-print" onclick="window.print()">
                            <i class="fas fa-print me-1"></i> Print Filtered Logs
                        </button>
                    </div>
                    <form class="row g-3 mb-4" method="get">
                        <div class="col-md-3">
                            <label for="teller" class="form-label">Teller</label>
                            <select name="teller" id="teller" class="form-select">
                                <option value="">All Tellers</option>
                                <?php foreach ($tellers as $teller): ?>
                                    <option value="<?php echo $teller['id']; ?>" <?php if ($selected_teller == $teller['id']) echo 'selected'; ?>>
                                        <?php echo htmlspecialchars($teller['full_name'] ?: $teller['username']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="date" name="start_date" id="start_date" class="form-control" value="<?php echo htmlspecialchars($start_date); ?>">
                        </div>
                        <div class="col-md-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="date" name="end_date" id="end_date" class="form-control" value="<?php echo htmlspecialchars($end_date); ?>">
                        </div>
                        <div class="col-md-3 align-self-end">
                            <button type="submit" class="btn btn-primary w-100"><i class="fas fa-search me-1"></i>Filter</button>
                        </div>
                    </form>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover align-middle mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>Date</th>
                                    <th>Teller</th>
                                    <th>Product</th>
                                    <th>Quantity</th>
                                    <th>Total</th>
                                    <th>Customer</th>
                                </tr>
                            </thead>
                            <tbody>
                            <?php if (empty($sales)): ?>
                                <tr><td colspan="6" class="text-center text-muted">No sales found for the selected criteria.</td></tr>
                            <?php else: ?>
                                <?php foreach ($sales as $sale): ?>
                                    <tr>
                                        <td><?php echo date('Y-m-d H:i', strtotime($sale['created_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($sale['full_name'] ?: $sale['username']); ?></td>
                                        <td><?php echo htmlspecialchars($sale['product_name']); ?></td>
                                        <td><?php echo (int)$sale['quantity']; ?></td>
                                        <td><?php echo number_format($sale['total_price'], 2); ?></td>
                                        <td><?php echo htmlspecialchars($sale['customer_name'] ?: 'Walk-in'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <?php include '../includes/footer.php'; ?>
        </div>
    </div>
</div>
</body>
</html> 